// $Id: CVstTempo.cpp,v 1.4 2007/02/08 21:08:23 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 *
 * Please note that VST is copyright Steinberg Media GmBh. No challenge is made to any of their trademarks
 * To use this file, you require a copy of the VST SDK, available from www.steinberg.net for free
 */

#include "CVstTempo.hpp"
using Exponent::Vst::CVstTempo;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CVstTempo, CCountedObject);

//	===========================================================================
CVstTempo::CVstTempo() : m_theEffect(NULL), m_lastTempoValid(false), m_songInformation(NULL)
{
	EXPONENT_CLASS_CONSTRUCTION(CVstTempo);

	// Null effect
	NULL_POINTER(m_theEffect);

	// New song info
	m_songInformation = new SSongInformation;

	// Default everything (assume host cannot send the info required)
	m_lastTempoValid = false;
	m_songInformation->initialise();
}

//	===========================================================================
CVstTempo::~CVstTempo()
{
	EXPONENT_CLASS_DESTRUCTION(CVstTempo);
	NULL_POINTER(m_theEffect);
	FREE_POINTER(m_songInformation);
}

//	===========================================================================
void CVstTempo::initialiseFromAudioEffectX(AudioEffectX *theEffect)
{
	if (theEffect == NULL)
	{
		throw CException("Effect is NULL", "CVstTempo::CVstTempo(AudioEffectX *)");
	}

	// Store the effect
	m_theEffect = theEffect;
}

//	===========================================================================
bool CVstTempo::getTempoInformation(const long flags)
{
	// Check if we have an actual effect
	if (m_theEffect == NULL)
	{
		throw CException("m_theEffect is NULL", "CVstTempo::getTempoInformation(const long *)");
	}

	// Get the time info
	VstTimeInfo *time = m_theEffect->getTimeInfo(flags);

	// Check if the time is null
	if (time == NULL)
	{
		m_lastTempoValid = false;
	}
	else
	{
		m_lastTempoValid = true;

		// Store the flags to be checked
		const long validFlags = time->flags;
		double tempoSeconds   = 0.0;

		// Store the transport information
		(validFlags & kVstTransportChanged) ? m_songInformation->m_transportChanged   = true : m_songInformation->m_transportChanged   = false;
		(validFlags & kVstTransportPlaying) ? m_songInformation->m_transportIsPlaying = true : m_songInformation->m_transportIsPlaying = false;

		// Get the time signature
		if (validFlags & kVstTempoValid)
		{
			m_songInformation->m_tempoIsValid		   = true;
			m_songInformation->m_samplesPerBeatIsValid = true;
			m_songInformation->m_bpm				   = time->tempo;
			tempoSeconds							   = time->tempo / 60.0;
			m_songInformation->m_samplesPerBeat		   = (long)(time->sampleRate / tempoSeconds);
		}
		else
		{
			m_songInformation->m_tempoIsValid		   = false;
			m_songInformation->m_samplesPerBeatIsValid = false;
			m_songInformation->m_bpm				   = 0.0;
			m_songInformation->m_samplesPerBeat		   = 0;
		}

		// Get the time signature
		if (validFlags & kVstTimeSigValid)
		{
			m_songInformation->m_timeSignatureIsValid  = true;
			m_songInformation->m_timeSignature.setTimeSignature(time->timeSigNumerator, time->timeSigDenominator);
		}
		else
		{
			m_songInformation->m_timeSignatureIsValid  = false;
			m_songInformation->m_timeSignature.setTimeSignature(4, 4);
		}

		// Get the ppq position
		if (validFlags & kVstPpqPosValid)
		{
			m_songInformation->m_ppqPositionIsValid = true;
			m_songInformation->m_ppqPosition        = time->ppqPos;
		}
		else
		{
			m_songInformation->m_ppqPositionIsValid = false;
			m_songInformation->m_ppqPosition        = 0.0;
		}

		// Store the bar start position
		if (validFlags & kVstBarsValid)
		{
			m_songInformation->m_barPositionIsValid = true;
			m_songInformation->m_barPosition = time->barStartPos;
		}
		else
		{
			m_songInformation->m_barPositionIsValid = false;
			m_songInformation->m_barPosition = 0.0;
		}

		// Store the bar position
		m_songInformation->m_samplePosition = time->samplePos;

		// Now compute the number of samples to the next bar
		if (m_songInformation->m_tempoIsValid && m_songInformation->m_ppqPositionIsValid && m_songInformation->m_barPositionIsValid && m_songInformation->m_timeSignatureIsValid)
        {
			double numberOfBeatsToBar = 0.0;
			if (time->barStartPos != time->ppqPos)
			{
				numberOfBeatsToBar = time->barStartPos + time->timeSigNumerator - time->ppqPos;

				// Whatch out for those buggy hosts
				while (numberOfBeatsToBar < 0.0)
				{
					numberOfBeatsToBar += time->timeSigNumerator;
				}
				while (numberOfBeatsToBar > time->timeSigNumerator)
				{
					numberOfBeatsToBar -= time->timeSigNumerator;
				}
			}

			// Compute the number of samples
			m_songInformation->m_samplesToNextBar = (long)(numberOfBeatsToBar * time->sampleRate / tempoSeconds);

			// Ensure we have a valid range
			if (m_songInformation->m_samplesToNextBar < 0)
            {
		        m_songInformation->m_samplesToNextBar = 0;
            }

			// We are valid and we are done ;)
			m_songInformation->m_samplesToNextBarIsValid = true;
        }
	}
	return m_lastTempoValid;
}